// System functions

#include "serial.c"
#include "ad_handler.c"
#include "widget_system_functions.c"
#include "functions_ok.c"
#include "functions_file.c"
#include "functions_core.c"
#include "functions_widget.c"
#include "functions_data_init.c"
#include "functions_measure.c"
#include "functions_peak.c"
#include "functions_print.c"
#include "functions_queue.c"

// LabChrom2 Init functions

extern int
Spawn (int Instr) {
pid_t		ChildPID;
char		*InstrNum[] = {"0", "1", "2", "3", "4"};

  if (PIDTable[Instr] != (pid_t)0) return -1;

  PRINTF ("Forking instrument %d, PID %d\n",Instr,getpid ());
  ChildPID = fork ();
  if (ChildPID != (pid_t)0) {
	// Parent
	PIDTable[Instr] = ChildPID;
	PRINTF ("Parent process, PID = %d\n", getpid ());
	PRINTF ("Child process inP, PID = %d\n", ChildPID);
  } else {
	// Child
	PRINTF ("Child process, PID = %d\n", getpid ());
	PRINTF ("ChildPID = %d\n", ChildPID);
	if (execlp ("labchrom2", "labchrom2", InstrNum[Instr], NULL) == -1) {
	  PRINTF ("***Error at execve of instrument %d\n", Instr);
	  exit (0);
	} // execve
  } // ChildPID
} // Spawn

extern int
Kill (int Instr) {
int KillR;
pid_t WaitR;
pid_t ToKill;

  if (PIDTable[Instr] == (pid_t)0) return -1;

  ToKill = PIDTable[Instr];
  KillR = kill (ToKill, SIGTERM);
  if (KillR != 0) printf ("***SIGTERM of instrument %d w/PID %d returned %d\n", Instr, ToKill, KillR);
  usleep (200000);
  KillR = kill (ToKill, SIGKILL);
  if (KillR != 0) printf ("***SIGKILL of instrument %d w/PID %d returned %d\n", Instr, ToKill, KillR);
  WaitR = waitpid (ToKill, NULL, 0);
  if (WaitR == -1) printf ("***Error at waitpid %d\n", errno);
} // Kill

extern void
AutoStart (void) {
int		i;
int		SpawnR;

  PIDTable[0] = getpid ();
  for (i = 1; i < MaxInstruments + 1; i++) {
	PIDTable[i] = 0;
  } // i
  for (i = 1; i < LC2.NumberOfInstruments + 1; i++) {
	if (LC2.Instruments[i].AutoStart == TRUE) SpawnR = Spawn (i);
  } // i
} // AutoStart

char
*ConfigName (int Instr, const char *GroupName, const char *FieldName) {
char	*TmpStr;

  strcpy (ConfigInternalName, "/labchrom");
  switch (Instr) {
	  case 0:
		  strcat (ConfigInternalName, "/");
		  break; // 0
	  case 1:
	  case 2:
	  case 3:
	  case 4:
		  TmpStr = calloc (10, sizeof(char));
		  sprintf (TmpStr, "%d/", Instr);
		  strcat (ConfigInternalName, TmpStr);
		  free (TmpStr);
		  break; // 1..4
	  default: {
		  strcat (ConfigInternalName, "99/");
	  } // default
  } // switch Instr
  strcat (ConfigInternalName, GroupName);
  strcat (ConfigInternalName, "/");
  strcat (ConfigInternalName, FieldName);
  return ConfigInternalName;
} // ConfigName

char
*FieldName (int Instr, const char *FieldName) {

  sprintf (ConfigInternalFieldName, "%s%d", FieldName, Instr);

  return ConfigInternalFieldName;
} // FieldName

void
ConfigString (char *DestStr, char *SourceStr) {
  if (SourceStr == NULL) {
	  strcpy (DestStr, " ");
	  PRINTF ("ConfigString SourceStr == NULL\n", 0);
  } // NULL
  else if (strlen (SourceStr) == 0) {
	  strcpy (DestStr, " ");
	  PRINTF ("ConfigString strlen (SourceStr) == 0\n", 0);
  } // 0
  else
  strcpy (DestStr, SourceStr);

  g_free (SourceStr);
} // ConfigString

extern void
ImportConfigLoad (int Instr) {
int DefaultFlag;

  ConfigString (ImportLastDir, gnome_config_get_string_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportLastDir")), &DefaultFlag));
  if (DefaultFlag) strcpy (ImportLastDir, "~/");
  ConfigString (ImportComment, gnome_config_get_string_with_default  (ConfigName(Instr, "Import", FieldName (Instr, "ImportComment")), &DefaultFlag));
  if (DefaultFlag) strcpy (ImportComment, "-");
  ImportChannels = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportChannels")), &DefaultFlag);
  if (DefaultFlag) ImportChannels = 1;
  ImportChannelsOf = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportChannelsOf")), &DefaultFlag);
  if (DefaultFlag) ImportChannelsOf = 4;
  ImportXResolution = gnome_config_get_float_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportXResolution")), &DefaultFlag);
  if (DefaultFlag) ImportXResolution = 1.0;
  ImportXAxisMenuIndex = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportXAxisMenuIndex")), &DefaultFlag);
  if (DefaultFlag) ImportXAxisMenuIndex = 0;
  ImportDecimation = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportDecimation")), &DefaultFlag);
  if (DefaultFlag) ImportDecimation = 1;
  ImportRevert = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportRevert")), &DefaultFlag);
  if (DefaultFlag) ImportRevert = FALSE;
  ImportScaleFactor = gnome_config_get_float_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportScaleFactor")), &DefaultFlag);
  if (DefaultFlag) ImportScaleFactor = 1000.0;
  ConfigString (ImportScaleDim, gnome_config_get_string_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportScaleDim")), &DefaultFlag));
  if (DefaultFlag) strcpy (ImportScaleDim, "mV");
  ImportResolution = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportResolution")), &DefaultFlag);
  if (DefaultFlag) ImportResolution = 1;
  ImportResolutionMenuIndex = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportResolutionMenuIndex")), &DefaultFlag);
  if (DefaultFlag) ImportResolutionMenuIndex = 0;
  ImportStoreSettings = gnome_config_get_int_with_default (ConfigName(Instr, "Import", FieldName (Instr, "ImportStoreSettings")), &DefaultFlag);
  if (DefaultFlag) ImportStoreSettings = FALSE;
} // ImportConfigLoad

extern void
ImportConfigSave (int Instr) {
  gnome_config_set_string (ConfigName(Instr, "Import", FieldName (Instr, "ImportLastDir")), ImportLastDir);
  gnome_config_set_string (ConfigName(Instr, "Import", FieldName (Instr, "ImportComment")), ImportComment);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportChannels")), ImportChannels);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportChannelsOf")), ImportChannelsOf);
  gnome_config_set_float (ConfigName(Instr, "Import", FieldName (Instr, "ImportXResolution")), ImportXResolution);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportXAxisMenuIndex")), ImportXAxisMenuIndex);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportDecimation")), ImportDecimation);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportRevert")), ImportRevert);
  gnome_config_set_float (ConfigName(Instr, "Import", FieldName (Instr, "ImportScaleFactor")), ImportScaleFactor);
  gnome_config_set_string (ConfigName(Instr, "Import", FieldName (Instr, "ImportScaleDim")), ImportScaleDim);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportResolution")), ImportResolution);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportResolutionMenuIndex")), ImportResolutionMenuIndex);
  gnome_config_set_int (ConfigName(Instr, "Import", FieldName (Instr, "ImportStoreSettings")), ImportStoreSettings);

  gnome_config_sync();
} // ImportConfigSave

extern void
ConfigLoad (int Instr) {
int ExistCheck;
int i,j,k;
int DefaultFlag;

  printf("Trying to load LabChrom configuration settings...\n");

  ExistCheck=gnome_config_get_int("/labchrom/ExistCheck/IfExists");
  if (ExistCheck!=1)
	{
	ConfigSave(0);
	ConfigSave(Instr);
	gnome_config_sync();
	printf("LabChrom system configuration settings has been rebuilt.\n");
	} // ExistCheck!=1

  CentralOptions.CentralWindowIconify = gnome_config_get_bool (ConfigName(0, "Central", "CentralWindowIconify"));
  CentralOptions.CentralWindowClose = gnome_config_get_bool (ConfigName(0, "Central", "CentralWindowClose"));

  LC2.NumberOfInstruments = gnome_config_get_int (ConfigName(0, "Central", "NumberOfInstruments"));
  for (i = 1; i<=MaxInstruments; i++) {
	LC2.Instruments[i].InstrumentType = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "InstrumentType")));
	LC2.Instruments[i].NumOfInstrument = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "NumOfInstrument")));
	LC2.Instruments[i].AutoStart = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "AutoStart")));
	ConfigString (LC2.Instruments[i].Instrument[0].InstrumentName, gnome_config_get_string (ConfigName(0, "Central", FieldName (i, "InstrumentName"))));
	LC2.Instruments[i].Instrument[0].ChromatographType = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ChromatographType")));
	LC2.Instruments[i].Instrument[0].SamplingMode = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "SamplingMode")));
	LC2.Instruments[i].Instrument[0].NumDetectors = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "NumDetectors")));
	ConfigString (LC2.Instruments[i].Instrument[0].IconForInstrument, gnome_config_get_string (ConfigName(0, "Central", FieldName (i, "IconForInstrument"))));

	for (j = 0; j < MaxDetectors; j++) {
	  ConfigString (LC2.Instruments[i].Instrument[0].Detectors[j].Name, gnome_config_get_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorName"))));
	  ConfigString (LC2.Instruments[i].Instrument[0].Detectors[j].Ext, gnome_config_get_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorExt"))));
	  LC2.Instruments[i].Instrument[0].Detectors[j].BindMainChannel = gnome_config_get_int (ConfigName(0, "Central", FieldName (10*i+j, "DetectorMainChannel")));
	  LC2.Instruments[i].Instrument[0].Detectors[j].BindSubChannel = gnome_config_get_int (ConfigName(0, "Central", FieldName (10*i+j, "DetectorSubChannel")));
	  LC2.Instruments[i].Instrument[0].Detectors[j].ScaleValue = gnome_config_get_float (ConfigName(0, "Central", FieldName (10*i+j, "DetectorScaleValue")));
	  ConfigString (LC2.Instruments[i].Instrument[0].Detectors[j].ScaleUnit, gnome_config_get_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorScaleUnit"))));
	} // j - MaxDetectors

	LC2.Instruments[i].ADParameters.ADDeviceId = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADDeviceId")));
	ConfigString (LC2.Instruments[i].ADParameters.ADDeviceName, gnome_config_get_string (ConfigName(0, "Central", FieldName (i, "ADDeviceName"))));
	LC2.Instruments[i].ADParameters.ADIOPortMenuIndex = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADIOPortMenuIndex")));
	ConfigString (LC2.Instruments[i].ADParameters.ADIOPortName, gnome_config_get_string (ConfigName(0, "Central", FieldName (i, "ADIOPortName"))));
	LC2.Instruments[i].ADParameters.ADNoHWCommunication = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADNoHWCommunication")));
	LC2.Instruments[i].ADParameters.ADMainChannelNum = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADMainChannelNum")));
	LC2.Instruments[i].ADParameters.ADSubChannelNum = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADSubChannelNum")));
	LC2.Instruments[i].ADParameters.ADChannelMode = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADChannelMode")));
	LC2.Instruments[i].ADParameters.ADFullScale = gnome_config_get_float (ConfigName(0, "Central", FieldName (i, "ADFullScale")));
	LC2.Instruments[i].ADParameters.ADResolution = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADResolution")));
	LC2.Instruments[i].ADParameters.ADZeroPoint = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADZeroPoint")));
	LC2.Instruments[i].ADParameters.ADBaseSamplingTime = gnome_config_get_float (ConfigName(0, "Central", FieldName (i, "ADBaseSamplingTime")));
	LC2.Instruments[i].ADParameters.ADBaseSamplingFrequencyInHz = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADBaseSamplingFrequencyInHz")));
	LC2.Instruments[i].ADParameters.ADSamplingSelection = gnome_config_get_int (ConfigName(0, "Central", FieldName (i, "ADSamplingSelection")));
	for (j = 0; j < MaxADSamplingArray; j++) {
	  LC2.Instruments[i].ADParameters.ADSamplingTimeTable[j] = gnome_config_get_float (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingTimeTable")));
	  LC2.Instruments[i].ADParameters.ADSamplingFrequencyTable[j] = gnome_config_get_int (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingFrequencyTable")));
	  LC2.Instruments[i].ADParameters.ADSamplingAccumulationTable[j] = gnome_config_get_int (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingAccumulationTable")));
	} // j - MaxADSamplingArray

	for (k = 0; k < MaxInstrument; k++) {
		for (j = 0; j < ControlSignalNum; j++) {
		  LC2.Instruments[i].Instrument[k].ControlSignals[j].Value = gnome_config_get_int (ConfigName(0, "Central", FieldName (1000*i+100*k+j, "ControlSignalValue")));
		  LC2.Instruments[i].Instrument[k].ControlSignals[j].ActiveLevel = gnome_config_get_int (ConfigName(0, "Central", FieldName (1000*i+100*k+j, "ControlSignalActiveLevel")));
// printf ("i = %d,   ControlSignals[%d].Value = %.2X\n", i, j, LC2.Instruments[i].Instrument[k].ControlSignals[j].Value);
		} // j - ControlSignalNum
	} // k - MaxInstrument
  } // i - MaxInstruments
  
  CentralOptions.PrintMarginLeft = gnome_config_get_float (ConfigName(0, "Print", "PrintMarginLeft"));
  CentralOptions.PrintMarginRight = gnome_config_get_float (ConfigName(0, "Print", "PrintMarginRight"));
  CentralOptions.PrintMarginTop = gnome_config_get_float (ConfigName(0, "Print", "PrintMarginTop"));
  CentralOptions.PrintMarginBottom = gnome_config_get_float (ConfigName(0, "Print", "PrintMarginBottom"));
  
  ConfigString (CentralOptions.PrintFontDefaultName, gnome_config_get_string (ConfigName(0, "Print", "PrintFontDefault")));
  ConfigString (CentralOptions.PrintFontTestName, gnome_config_get_string (ConfigName(0, "Print", "PrintFontTest")));
  ConfigString (CentralOptions.PrintFontChromPeakDataName, gnome_config_get_string (ConfigName(0, "Print", "PrintFontChromPeakData")));
  ConfigString (CentralOptions.PrintFontPeakListName, gnome_config_get_string (ConfigName(0, "Print", "PrintFontPeakList")));
  ConfigString (CentralOptions.PrintFontAxisName, gnome_config_get_string (ConfigName(0, "Print", "PrintFontAxis")));
  
  ConfigString (BugTraceCentralDir, gnome_config_get_string (ConfigName(0, "Directories", "BugTraceCentralDir")));
  
  if (Instr == 0) return;

  /*
  ConfigString (DestStr, gnome_config_get_string (ConfigName(Instr, "Group", "FieldName")));
  DestInt = gnome_config_get_int (ConfigName(Instr, "Group", "FieldName"));
  DestFloat = gnome_config_get_float (ConfigName(Instr, "Group", "FieldName"));
  */
  
  // Common variables
  
  ConfigString (BugTraceDir, gnome_config_get_string (ConfigName(Instr, "Directories", "BugTraceDir")));
  
  // General
  MainWindowShrink = gnome_config_get_int (ConfigName(Instr, "General", "MainWindowShrink"));
  MainWindowKeepOnTop = gnome_config_get_int (ConfigName(Instr, "General", "MainWindowKeepOnTop"));
  
//  ConfigString (LastSampleQueue, gnome_config_get_string (ConfigName(Instr, "LastFiles", "LastSampleQueue")));
//  ConfigString (LastChrom, gnome_config_get_string (ConfigName(Instr, "LastFiles", "LastChrom")));
//  ConfigString (LastCalibr, gnome_config_get_string (ConfigName(Instr, "LastFiles", "LastCalibr")));
  
  for (i = 0; i < MaxInstrument; i++) {
	  Options[i].DataAcqMode = gnome_config_get_int (ConfigName(Instr, "MeasureSettings", FieldName (i, "DataAcqMode")));
	  // Common
	  ConfigString (Options[i].MeasureDataDir, gnome_config_get_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureDataDir"))));
	  Options[i].GenerateWorkDir = gnome_config_get_int (ConfigName(Instr, "MeasureCommon", FieldName (i, "GenerateWorkDir")));
	  ConfigString (Options[i].MeasureMethod, gnome_config_get_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureMethod"))));
	  ConfigString (Options[i].MeasureFilename, gnome_config_get_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureFilename"))));
	  // Unique
	  Options[i].RegisterInQueue = gnome_config_get_int (ConfigName(Instr, "MeasureUnique", FieldName (i, "RegisterInQueue")));
	  // Queue
	  ConfigString (Options[i].QueueName, gnome_config_get_string (ConfigName(Instr, "MeasureQueue", FieldName (i, "QueueName"))));
	  Options[i].StoreDefaults = gnome_config_get_int (ConfigName(Instr, "MeasureQueue", FieldName (i, "StoreDefaults")));
	  // Continous
	  Options[i].LimitedNumber = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "LimitedNumber")));
	  Options[i].LimitedNumOfChrom = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "LimitedNumOfChrom")));
	  Options[i].RestartMode = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartMode")));
	  Options[i].RestartChrom = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartChrom")));
	  Options[i].RestartTime = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartTime")));
	  Options[i].UseTemplateQueue = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "UseTemplateQueue")));
	  ConfigString (Options[i].TemplateQueueName, gnome_config_get_string (ConfigName(Instr, "MeasureContinous", FieldName (i, "TemplateQueueName"))));
	  Options[i].AutoGenerateDir = gnome_config_get_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "AutoGenerateDir")));
	  // Directories
	  ConfigString (Options[i].DefaultDataRootDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "DefaultDataRootDir"))));
	  ConfigString (Options[i].ActiveDataDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "ActiveDataDir"))));
	  ConfigString (Options[i].MethodDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "MethodDir"))));
	  ConfigString (Options[i].CalibrDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "CalibrDir"))));
	  ConfigString (Options[i].SpecificDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "SpecificDir"))));
	  ConfigString (Options[i].ImportDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "ImportDir"))));
	  ConfigString (Options[i].ExportDir, gnome_config_get_string (ConfigName(Instr, "Directories", FieldName (i, "ExportDir"))));
	  // Other
	  ConfigString (Options[i].LastName, gnome_config_get_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastName"))));
	  ConfigString (Options[i].LastDataDir, gnome_config_get_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastDataDir"))));
	  ConfigString (Options[i].LastMethod, gnome_config_get_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastMethod"))));
	  ConfigString (Options[i].LastCalibr, gnome_config_get_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastCalibr"))));
	  // Print
	  Options[i].PrintFrame = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintFrame")));
	  Options[i].PrintTickMarks = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintTickMarks")));
	  Options[i].PrintBaselines = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintBaselines")));
	  Options[i].PrintXAxisTicks = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintXAxisTicks")));
	  Options[i].PrintXAxisValues = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintXAxisValues")));
	  Options[i].PrintPeakName = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakName")));
	  Options[i].PrintPeakNumber = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakNumber")));
	  Options[i].PrintPeakRetention = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakRetention")));
	  Options[i].PrintPeakConcentration = gnome_config_get_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakConcentration")));
	  ConfigString (Options[i].PrintTemplate, gnome_config_get_string (ConfigName(Instr, "Print", FieldName (i, "PrintTemplate"))));
	  // Export
	  Options[i].ExportMode = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportMode")));
	  Options[i].ExportDOSLineFeed = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportDOSLineFeed")));
	  Options[i].ExportExtendedHeader = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportExtendedHeader")));
	  Options[i].ExportQuantitativeData = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportQuantitativeData")));
	  Options[i].ExportSampleInformation = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportSampleInformation")));
	  Options[i].ExportInstrumentParams = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportInstrumentParams")));
	  Options[i].ExportDataInTable = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportDataInTable")));
	  Options[i].ExportSignatures = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportSignatures")));
	  ConfigString (Options[i].ExportTemplate, gnome_config_get_string (ConfigName(Instr, "Export", FieldName (i, "ExportTemplate"))));
	  ConfigString (Options[i].ExportInternalTemplate, gnome_config_get_string (ConfigName(Instr, "Export", FieldName (i, "ExportInternalTemplate"))));
	  Options[i].ExportInputEnc = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportInputEnc")));
	  ConfigString (Options[i].ExportInputEncOther, gnome_config_get_string (ConfigName(Instr, "Export", FieldName (i, "ExportInputEncOther"))));
	  Options[i].ExportOutputEnc = gnome_config_get_int (ConfigName(Instr, "Export", FieldName (i, "ExportOutputEnc")));
	  ConfigString (Options[i].ExportOutputEncOther, gnome_config_get_string (ConfigName(Instr, "Export", FieldName (i, "ExportOutputEncOther"))));
	  
	  for (j = 0; j < MaxDetectors; j++) {
		  Options[i].MeasureAttenuator[j] = gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "WorkstationAttenuator"))));
		  Options[i].MeasureShift[j] = gnome_config_get_float (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "WorkstationShift"))));
		  Options[i].red[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_red"))));
		  Options[i].green[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_green"))));
		  Options[i].blue[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_blue"))));
	  } // j 
	  Options[i].MeasureAxisTime = gnome_config_get_float (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAxisTime")));
	  Options[i].MeasureAxisValue = gnome_config_get_float (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAxisValue")));
	  Options[i].ShowAllDetectors = gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAllDet")));
	  Options[i].ShowMeasureFilenames = gnome_config_get_int (ConfigName(Instr, "Workstation", FieldName (i, "ShowMeasureFilenames")));
  } // i
  
  serial_no_communicate = LC2.Instruments[Instr].ADParameters.ADNoHWCommunication;
  //serial_no_communicate = TRUE;
  
  /*
  	Instrument specific variables
  */
  
  // Workstation

  // Datastation
  
  // TR541
  
  // TR542
  
  // TR930
  
  // General variables
  ReprocessShowLists = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowLists")));
  ReprocessShowQueue = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowQueue")));
  ReprocessShowOverlay = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowOverlay")));
  ReprocessShowTickMarks = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowTickMarks")));
  ReprocessShowBaselines = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowBaselines")));
  ReprocessShowFillPeaks = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowFillPeaks")));
  ReprocessShowPeakNumbers = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowPeakNumbers")));
  ReprocessShowRetention = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowRetention")));
  ReprocessShowName = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowName")));
  ReprocessShowComment = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowComment")));
  ReprocessShowTime = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowTime")));
  Options[WorkArea].ShowAllDetectors = gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowAllDet")));
  for (j = 0; j < MaxDetectors; j++) {
	  Options[WorkArea].red[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_red"))));
	  Options[WorkArea].green[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_green"))));
	  Options[WorkArea].blue[j] = (guint)gnome_config_get_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_blue"))));
  } // j 
  
  for (j = 0; j < MaxOverlayedChrom; j++) {
	  OverlayColor[j].red = (guint)gnome_config_get_int (ConfigName(Instr, "OverlayColor", FieldName(j, "Overlay_red")));
	  OverlayColor[j].green = (guint)gnome_config_get_int (ConfigName(Instr, "OverlayColor", FieldName(j, "Overlay_green")));
	  OverlayColor[j].blue = (guint)gnome_config_get_int (ConfigName(Instr, "OverlayColor", FieldName(j, "Overlay_blue")));
  } // j 
  // Import functions
  ImportConfigLoad (Instr);
  
  // Other, work settings
  for (i = 0; i < MaxInstrument; i++)
	  Chrom[i].ActualExtension = gnome_config_get_int (ConfigName (Instr, "OtherWork", FieldName (i, "ActualExtension")));
  for (j = 0; j < MaxGradRatio; j++) {
	  GradientConfigColors[j].red = (guint)gnome_config_get_int (ConfigName(Instr, "Other", FieldName(j, "grad_red")));
	  GradientConfigColors[j].green = (guint)gnome_config_get_int (ConfigName(Instr, "Other", FieldName(j, "grad_green")));
	  GradientConfigColors[j].blue = (guint)gnome_config_get_int (ConfigName(Instr, "Other", FieldName(j, "grad_blue")));
  } // j 
} // ConfigLoad

extern void
ConfigCentralSave (void) {
int		i,j,k;

  gnome_config_set_bool (ConfigName(0, "Central", "CentralWindowIconify"), CentralOptions.CentralWindowIconify);
  gnome_config_set_bool (ConfigName(0, "Central", "CentralWindowClose"), CentralOptions.CentralWindowClose);

  gnome_config_set_int (ConfigName(0, "Central", "NumberOfInstruments"), LC2.NumberOfInstruments);
  for (i = 1; i<=MaxInstruments; i++) {
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "InstrumentType")), LC2.Instruments[i].InstrumentType);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "NumOfInstrument")), LC2.Instruments[i].NumOfInstrument);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "AutoStart")), LC2.Instruments[i].AutoStart);
	gnome_config_set_string (ConfigName(0, "Central", FieldName (i, "InstrumentName")), LC2.Instruments[i].Instrument[0].InstrumentName);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ChromatographType")), LC2.Instruments[i].Instrument[0].ChromatographType);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "SamplingMode")), LC2.Instruments[i].Instrument[0].SamplingMode);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "NumDetectors")), LC2.Instruments[i].Instrument[0].NumDetectors);
	gnome_config_set_string (ConfigName(0, "Central", FieldName (i, "IconForInstrument")), LC2.Instruments[i].Instrument[0].IconForInstrument);

	for (j = 0; j < MaxDetectors; j++) {
	  gnome_config_set_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorName")), LC2.Instruments[i].Instrument[0].Detectors[j].Name);
	  gnome_config_set_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorExt")), LC2.Instruments[i].Instrument[0].Detectors[j].Ext);
	  gnome_config_set_int (ConfigName(0, "Central", FieldName (10*i+j, "DetectorMainChannel")), LC2.Instruments[i].Instrument[0].Detectors[j].BindMainChannel);
	  gnome_config_set_int (ConfigName(0, "Central", FieldName (10*i+j, "DetectorSubChannel")), LC2.Instruments[i].Instrument[0].Detectors[j].BindSubChannel);
	  gnome_config_set_float (ConfigName(0, "Central", FieldName (10*i+j, "DetectorScaleValue")), LC2.Instruments[i].Instrument[0].Detectors[j].ScaleValue);
	  gnome_config_set_string (ConfigName(0, "Central", FieldName (10*i+j, "DetectorScaleUnit")), LC2.Instruments[i].Instrument[0].Detectors[j].ScaleUnit);
	} // j - MaxDetectors

	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADDeviceId")), LC2.Instruments[i].ADParameters.ADDeviceId);
	gnome_config_set_string (ConfigName(0, "Central", FieldName (i, "ADDeviceName")), LC2.Instruments[i].ADParameters.ADDeviceName);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADIOPortMenuIndex")), LC2.Instruments[i].ADParameters.ADIOPortMenuIndex);
	gnome_config_set_string (ConfigName(0, "Central", FieldName (i, "ADIOPortName")), LC2.Instruments[i].ADParameters.ADIOPortName);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADNoHWCommunication")), LC2.Instruments[i].ADParameters.ADNoHWCommunication);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADMainChannelNum")), LC2.Instruments[i].ADParameters.ADMainChannelNum);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADSubChannelNum")), LC2.Instruments[i].ADParameters.ADSubChannelNum);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADChannelMode")), LC2.Instruments[i].ADParameters.ADChannelMode);
	gnome_config_set_float (ConfigName(0, "Central", FieldName (i, "ADFullScale")), LC2.Instruments[i].ADParameters.ADFullScale);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADResolution")), LC2.Instruments[i].ADParameters.ADResolution);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADZeroPoint")), LC2.Instruments[i].ADParameters.ADZeroPoint);
	gnome_config_set_float (ConfigName(0, "Central", FieldName (i, "ADBaseSamplingTime")), LC2.Instruments[i].ADParameters.ADBaseSamplingTime);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADBaseSamplingFrequencyInHz")), LC2.Instruments[i].ADParameters.ADBaseSamplingFrequencyInHz);
	gnome_config_set_int (ConfigName(0, "Central", FieldName (i, "ADSamplingSelection")), LC2.Instruments[i].ADParameters.ADSamplingSelection);
	for (j = 0; j < MaxADSamplingArray; j++) {
	  gnome_config_set_float (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingTimeTable")), LC2.Instruments[i].ADParameters.ADSamplingTimeTable[j]);
	  gnome_config_set_int (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingFrequencyTable")), LC2.Instruments[i].ADParameters.ADSamplingFrequencyTable[j]);
	  gnome_config_set_int (ConfigName(0, "Central", FieldName (10*i+j, "ADSamplingAccumulationTable")), LC2.Instruments[i].ADParameters.ADSamplingAccumulationTable[j]);
	} // j

	for (k = 0; k < MaxInstrument; k++) {
		for (j = 0; j < ControlSignalNum; j++) {
		  gnome_config_set_int (ConfigName(0, "Central", FieldName (1000*i+100*k+j, "ControlSignalValue")), LC2.Instruments[i].Instrument[k].ControlSignals[j].Value);
		  gnome_config_set_int (ConfigName(0, "Central", FieldName (1000*i+100*k+j, "ControlSignalActiveLevel")), LC2.Instruments[i].Instrument[k].ControlSignals[j].ActiveLevel);
		} // j - ControlSignalNum
	} // k - MaxInstrument
  } // i - MaxInstruments

  gnome_config_set_float (ConfigName(0, "Print", "PrintMarginLeft"), CentralOptions.PrintMarginLeft);
  gnome_config_set_float (ConfigName(0, "Print", "PrintMarginRight"), CentralOptions.PrintMarginRight);
  gnome_config_set_float (ConfigName(0, "Print", "PrintMarginTop"), CentralOptions.PrintMarginTop);
  gnome_config_set_float (ConfigName(0, "Print", "PrintMarginBottom"), CentralOptions.PrintMarginBottom);
  
  gnome_config_set_string (ConfigName(0, "Print", "PrintFontDefault"), CentralOptions.PrintFontDefaultName);
  gnome_config_set_string (ConfigName(0, "Print", "PrintFontTest"), CentralOptions.PrintFontTestName);
  gnome_config_set_string (ConfigName(0, "Print", "PrintFontChromPeakData"), CentralOptions.PrintFontChromPeakDataName);
  gnome_config_set_string (ConfigName(0, "Print", "PrintFontPeakList"), CentralOptions.PrintFontPeakListName);
  gnome_config_set_string (ConfigName(0, "Print", "PrintFontAxis"), CentralOptions.PrintFontAxisName);
  
  gnome_config_set_string (ConfigName(0, "Directories", "BugTraceCentralDir"), BugTraceCentralDir);
  
} // ConfigCentralSave

extern void
ConfigInstrumentSave (int Instr) {
int i,j;
  // Common variables
  
  gnome_config_set_string (ConfigName(Instr, "Directories", "BugTraceDir"), BugTraceDir);
  
  // General
  gnome_config_set_int (ConfigName(Instr, "General", "MainWindowShrink"), MainWindowShrink);
  gnome_config_set_int (ConfigName(Instr, "General", "MainWindowKeepOnTop"), MainWindowKeepOnTop);
  
//  gnome_config_set_string (ConfigName(Instr, "LastFiles", "LastSampleQueue"), LastSampleQueue);
//  gnome_config_set_string (ConfigName(Instr, "LastFiles", "LastChrom"), LastChrom);
//  gnome_config_set_string (ConfigName(Instr, "LastFiles", "LastCalibr"), LastCalibr);
  
  for (i = 0; i < MaxInstrument; i++) {
	  gnome_config_set_int (ConfigName(Instr, "MeasureSettings", FieldName (i, "DataAcqMode")), Options[i].DataAcqMode);
	  // Common
	  gnome_config_set_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureDataDir")), Options[i].MeasureDataDir);
	  gnome_config_set_int (ConfigName(Instr, "MeasureCommon", FieldName (i, "GenerateWorkDir")), Options[i].GenerateWorkDir);
	  gnome_config_set_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureMethod")), Options[i].MeasureMethod);
	  gnome_config_set_string (ConfigName(Instr, "MeasureCommon", FieldName (i, "MeasureFilename")), Options[i].MeasureFilename);
	  // Unique
	  gnome_config_set_int (ConfigName(Instr, "MeasureUnique", FieldName (i, "RegisterInQueue")), Options[i].RegisterInQueue);
	  // Queue
	  gnome_config_set_string (ConfigName(Instr, "MeasureQueue", FieldName (i, "QueueName")), Options[i].QueueName);
	  gnome_config_set_int (ConfigName(Instr, "MeasureQueue", FieldName (i, "StoreDefaults")), Options[i].StoreDefaults);
	  // Continous
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "LimitedNumber")), Options[i].LimitedNumber);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "LimitedNumOfChrom")), Options[i].LimitedNumOfChrom);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartMode")), Options[i].RestartMode);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartChrom")), Options[i].RestartChrom);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "RestartTime")), Options[i].RestartTime);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "UseTemplateQueue")), Options[i].UseTemplateQueue);
	  gnome_config_set_string (ConfigName(Instr, "MeasureContinous", FieldName (i, "TemplateQueueName")), Options[i].TemplateQueueName);
	  gnome_config_set_int (ConfigName(Instr, "MeasureContinous", FieldName (i, "AutoGenerateDir")), Options[i].AutoGenerateDir);
	  // Directories
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "DefaultDataRootDir")), Options[i].DefaultDataRootDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "ActiveDataDir")), Options[i].ActiveDataDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "MethodDir")), Options[i].MethodDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "CalibrDir")), Options[i].CalibrDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "SpecificDir")), Options[i].SpecificDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "ImportDir")), Options[i].ImportDir);
	  gnome_config_set_string (ConfigName(Instr, "Directories", FieldName (i, "ExportDir")), Options[i].ExportDir);
	  // Other
	  gnome_config_set_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastName")), Options[i].LastName);
	  gnome_config_set_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastDataDir")), Options[i].LastDataDir);
	  gnome_config_set_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastMethod")), Options[i].LastMethod);
	  gnome_config_set_string (ConfigName(Instr, "MeasureOther", FieldName (i, "LastCalibr")), Options[i].LastCalibr);
	  // Print
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintFrame")), Options[i].PrintFrame);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintTickMarks")), Options[i].PrintTickMarks);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintBaselines")), Options[i].PrintBaselines);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintXAxisTicks")), Options[i].PrintXAxisTicks);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintXAxisValues")), Options[i].PrintXAxisValues);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakName")), Options[i].PrintPeakName);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakNumber")), Options[i].PrintPeakNumber);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakRetention")), Options[i].PrintPeakRetention);
	  gnome_config_set_int (ConfigName(Instr, "Print", FieldName (i, "PrintPeakConcentration")), Options[i].PrintPeakConcentration);
	  gnome_config_set_string (ConfigName(Instr, "Print", FieldName (i, "PrintTemplate")), Options[i].PrintTemplate);
	  // Export
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportMode")), Options[i].ExportMode);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportDOSLineFeed")), Options[i].ExportDOSLineFeed);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportExtendedHeader")), Options[i].ExportExtendedHeader);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportQuantitativeData")), Options[i].ExportQuantitativeData);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportSampleInformation")), Options[i].ExportSampleInformation);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportInstrumentParams")), Options[i].ExportInstrumentParams);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportDataInTable")), Options[i].ExportDataInTable);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportSignatures")), Options[i].ExportSignatures);
	  gnome_config_set_string (ConfigName(Instr, "Export", FieldName (i, "ExportTemplate")), Options[i].ExportTemplate);
	  gnome_config_set_string (ConfigName(Instr, "Export", FieldName (i, "ExportInternalTemplate")), Options[i].ExportInternalTemplate);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportInputEnc")), Options[i].ExportInputEnc);
	  gnome_config_set_string (ConfigName(Instr, "Export", FieldName (i, "ExportInputEncOther")), Options[i].ExportInputEncOther);
	  gnome_config_set_int (ConfigName(Instr, "Export", FieldName (i, "ExportOutputEnc")), Options[i].ExportOutputEnc);
	  gnome_config_set_string (ConfigName(Instr, "Export", FieldName (i, "ExportOutputEncOther")), Options[i].ExportOutputEncOther);
	  
	  for (j = 0; j < MaxDetectors; j++) {
		  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "WorkstationAttenuator"))), Options[i].MeasureAttenuator[j]);
		  gnome_config_set_float (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "WorkstationShift"))), Options[i].MeasureShift[j]);
		  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_red"))), Options[i].red[j]);
		  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_green"))), Options[i].green[j]);
		  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, FieldName(j, "Workstation_blue"))), Options[i].blue[j]);
	  } // j 
	  gnome_config_set_float (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAxisTime")), Options[i].MeasureAxisTime);
	  gnome_config_set_float (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAxisValue")), Options[i].MeasureAxisValue);
	  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, "WorkstationAllDet")), Options[i].ShowAllDetectors);
	  gnome_config_set_int (ConfigName(Instr, "Workstation", FieldName (i, "ShowMeasureFilenames")), Options[i].ShowMeasureFilenames);
  } // i
  
  // Workstation
  
  // Datastation
  
  // TR541
  
  // TR542
  
  // TR930
  
  // General variables
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowLists")), ReprocessShowLists);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowQueue")), ReprocessShowQueue);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowOverlay")), ReprocessShowOverlay);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowTickMarks")), ReprocessShowTickMarks);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowBaselines")), ReprocessShowBaselines);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowFillPeaks")), ReprocessShowFillPeaks);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowPeakNumbers")), ReprocessShowPeakNumbers);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowRetention")), ReprocessShowRetention);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowName")), ReprocessShowName);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowComment")), ReprocessShowComment);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowTime")), ReprocessShowTime);
  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, "ReprocessShowAllDet")), Options[WorkArea].ShowAllDetectors);
  for (j = 0; j < MaxDetectors; j++) {
	  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_red"))), Options[WorkArea].red[j]);
	  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_green"))), Options[WorkArea].green[j]);
	  gnome_config_set_int (ConfigName(Instr, "Reprocess", FieldName (Instr, FieldName(j, "Reprocess_blue"))), Options[WorkArea].blue[j]);
  } // j 
  
  for (j = 0; j < MaxOverlayedChrom; j++) {
	  gnome_config_set_int (ConfigName(Instr, "OverlayColor", FieldName (j, "Overlay_red")), OverlayColor[j].red);
	  gnome_config_set_int (ConfigName(Instr, "OverlayColor", FieldName (j, "Overlay_green")), OverlayColor[j].green);
	  gnome_config_set_int (ConfigName(Instr, "OverlayColor", FieldName (j, "Overlay_blue")), OverlayColor[j].blue);
  } // j 
  
  // Import functions
  if (ImportStoreSettings != FALSE) ImportConfigSave (Instr);
  
  // Other, work settings
  for (i = 0; i < MaxInstrument; i++)
	  gnome_config_set_int (ConfigName (Instr, "OtherWork", FieldName (i, "ActualExtension")), Chrom[i].ActualExtension);
  for (j = 0; j < MaxGradRatio; j++) {
	  gnome_config_set_int (ConfigName(Instr, "Other", FieldName(j, "grad_red")), GradientColors[j].red);
	  gnome_config_set_int (ConfigName(Instr, "Other", FieldName(j, "grad_green")), GradientColors[j].green);
	  gnome_config_set_int (ConfigName(Instr, "Other", FieldName(j, "grad_blue")), GradientColors[j].blue);
  } // j 
} // ConfigInstrumentSave

extern void
ConfigSave (int Instr) {
BT_INIT();

  sprintf (BugTraceUserString, "ConfigSave executed of instrument %d\n", Instr);
  BUGTRACE (BugTraceUserString);
  gnome_config_set_int("/labchrom/ExistCheck/IfExists",1);
  if (Instr == 0)
	  ConfigCentralSave ();
  else
	  ConfigInstrumentSave (Instr);
  gnome_config_sync();
} // ConfigSave

extern void
ShrinkMainWindow (void) {
GtkWidget		*MainHide;

  if (MainWindowShrinked) return;
  if (OverrideMainWindowShrink) return;

  MainHide = lookup_widget (MainWindow, "MainHPaned");
  gtk_widget_hide (MainHide);
  MainHide = lookup_widget (MainWindow, "MainAppBar");
  gtk_widget_hide (MainHide);

  gtk_window_resize (GTK_WINDOW(MainWindow), 1, 1);
  if (MainWindowKeepOnTop)
	  gtk_window_set_keep_above (GTK_WINDOW(MainWindow), TRUE);

  MainWindowShrinked = TRUE;
  printf ("ShrinkMainWindow\n");
} // ShrinkMainWindow

extern void
ExpandMainWindow (void) {
GtkWidget		*MainHide;

  if (!MainWindowShrinked) return;

  MainHide = lookup_widget (MainWindow, "MainHPaned");
  gtk_widget_show (MainHide);
  MainHide = lookup_widget (MainWindow, "MainAppBar");
  gtk_widget_show (MainHide);

  gtk_window_set_keep_above (GTK_WINDOW(MainWindow), FALSE);
  
  MainWindowShrinked = FALSE;
  printf ("ExpandMainWindow\n");
} // ExpandMainWindow

extern void
LabChromInit (void) {
int i,j;

  strcpy (LabChromID, "WFOBXXGCABDM74980");
  strcpy (LabChromText, "LabChrom - Chemotron Inc. 1991-2004");

  for (i = 1; i < FlagMax; i++) {
	  for (j = 0; j < MaxChromData; j++) {
		  Chrom[j].FileLoaded[i] = FALSE;
		  Chrom[j].FileSaved[i] = FALSE;
	  } // j
  } // i

  NewSpectra (&Spectra);

  WorkAD = LC2.Instruments[ActualInstrument].ADParameters;
  for (i = 0; i < MaxInstrument; i++) {
	  WorkInstrument[i] = LC2.Instruments[ActualInstrument].Instrument[i];
  } // i
  WorkInstruments = LC2.Instruments[ActualInstrument];

  // Generic measure thread
  MeasureInitFunction = DefaultInitFunction;
  MeasureDoneFunction = DefaultDoneFunction;
  MeasureCommandFunction = DefaultCommandFunction;
  MeasureExecFunction = DefaultExecFunction;
  MeasureErrorFunction = DefaultErrorFunction;

  switch (LC2.Instruments[ActualInstrument].InstrumentType) {
	case InstrumentNone :
		break;
	case InstrumentWorkstation :
	case InstrumentDataStation :
		  for (i = 0; i < MaxInstrument; i++) {
			  if ((Options[i].MeasureAxisTime < 5) || (Options[0].MeasureAxisTime > 120))
				  Options[i].MeasureAxisTime = 5;
			  Options[i].MeasureAxisValue = 1.0;
		  } // i
		  MeasureCommandFunction = GenericCommandFunction;
		  MeasureExecFunction = GenericDataAcqExecFunction;
		  MeasureErrorFunction = GenericErrorFunction;
		break;
	case InstrumentTR541 :
		  MeasureCommandFunction = GenericCommandFunction;
		  MeasureExecFunction = TR541MeasureExecFunction;
		  MeasureErrorFunction = GenericErrorFunction;
		break;
	case InstrumentTR542 :
		break;
	case InstrumentTR930 :
		break;
  } // switch

  EventListInit ();
  ReprocessShowListsReady = FALSE;
  ReprocessOverlayChromNum = 0;
  ReprocessOverlaySelected = -1;
  ReprocessOverlayCListSelected = -1;
  ReprocessOverlayDrawMode = 0;
  ReprocessOverlayFunction = 0;
  for (i = 0; i < MaxOverlayedChrom; i++) {
	  ReprocessOverlayShift[i] = 0;
	  ReprocessOverlayZoom[i] = 1.0;
	  ReprocessOverlayShow[i] = TRUE;
	  ReprocessOverlayEnable[i] = TRUE;
  } // i
  SelectedPeak = -1;
  strcpy (LastChromDir, Options[0].DefaultDataRootDir);
} // LabChromInit

extern void
LabChromDone (void) {
  switch (LC2.Instruments[ActualInstrument].InstrumentType) {
	case InstrumentNone :
		break;
	case InstrumentWorkstation :
		break;
	case InstrumentDataStation :
		break;
	case InstrumentTR541 :
		break;
	case InstrumentTR542 :
		break;
	case InstrumentTR930 :
		break;
  } // switch
} // LabChromDone

extern void
MeasureInit (void) {
int i,j,k;
  if (ActualInstrument > 0) {
	  for (i = 0; i < MaxInstrument; i++) {
		  MeasureTickCounter[i] = 0;
		  MeasureDataCounter[i] = 0;
		  MeasureFreshDataFlag[i] = 0;
		  for (j = 0; j < MaxDetectors; j++) {
			  MeasureDataValue[i][j] = 0;
			  MeasureScaleValue[i][j] = 0;
		  } // j
		  for (j = 0; j < MaxGradRatio; j++) {
			  MeasureGradRatio[i][j] = 0;
		  } // j
		  NewGradValue[i] = FALSE;
		  NewGradCheck[i] = FALSE;
		  GradCounter[i] = 0;
		  MeasureResponseValue[i] = 0;
		  MeasureErrorFlag[i] = 0;
	  } // i
	  for (i = 0; i < MaxMainChannels; i++) {
		  for (j = 0; j < MaxSubChannels; j++)
			  ADRawValue[i][j] = 0;
		  for (j = 0; j < MaxGradRatio; j++)
			  ADRawGradRatio[i][j] = 0;
		  ADRawResponse[i] = 0;
		  ADRawFreshData[i] = 0;
		  ADRawINPort[i] = 0;
		  ADRawOUTPort[i] = 0;
		  ADRawTimeOut[i] = 0;
		  ADFreshOUTFlag[i] = FALSE;
		  ADFreshDAFlag[i] = FALSE;
	  } // i
	  for (i = 0; i < MaxInstrument; i++) {
		  for (k = 0; k < ControlSignalNum; k++) {
			  if (WorkInstruments.Instrument[i].ControlSignals[k].ActiveLevel) {
				  ADRawOUTPort[i] |= WorkInstruments.Instrument[i].ControlSignals[k].Value;
				  //printf ("Instr: %d,   Signal #: %d,   Signal: %.2X,   Signal SET: %.2X\n", i, k, WorkInstruments.Instrument[i].ControlSignals[k].Value, ADRawOUTPort[i]);
			  } // if Level
		  } // k
	  } // i
	  ADInit ();
	  pthread_mutex_init (&measure_data_mutex, NULL);
	  pthread_mutex_init (&measure_parameter_mutex, NULL);
	  pthread_mutex_init (&measure_command_mutex, NULL);
	  // Join measure thread
	  pthread_create (&measure_thread, NULL, &generic_measure_function, NULL);
  } // ActualInstrument > 0
} // MeasureInit

extern void
MeasureDone (void) {
int	TECounter = 0;

  if (ActualInstrument > 0) {
	  ThreadExit = TRUE;
	  while (ThreadExitOk == FALSE) {
		  TECounter++;
		  if (TECounter == 10) break;
		  usleep (50000);
	  } // while
	  pthread_cancel (measure_thread);
	  pthread_join (measure_thread, NULL);
	  pthread_mutex_destroy (&measure_parameter_mutex);
	  pthread_mutex_destroy (&measure_command_mutex);
	  pthread_mutex_destroy (&measure_data_mutex);
	  ADClose ();
  } // ActualInstrument > 0
} // MeasureDone

extern void
SystemInit (void) {
int i,j;

  NetCDFxml = NULL;

  MainWindowShrink = FALSE;
  MainWindowShrinked = FALSE;
  MainWindowKeepOnTop = FALSE;
  OverrideMainWindowShrink = FALSE;

  ConfigInternalName = calloc (FilenameLength, sizeof (char));
  ConfigInternalFieldName = calloc (FilenameLength, sizeof (char));
  ASCIITime = calloc (100, sizeof (char));

  BugTraceDir = calloc (FilenameLength, sizeof (char));
  BugTraceCentralDir = calloc (FilenameLength, sizeof (char));

  SpectraName = calloc (FilenameLength, sizeof (char));

  ImportLastDir = calloc (FilenameLength, sizeof (char));
  ImportComment = calloc (FilenameLength, sizeof (char));
  ImportScaleDim = calloc (100, sizeof (char));

  CentralOptions.CentralWindowIconify = FALSE;
  MainDrawingAreaNeedRedraw = FALSE;

  MainThreadLock = FALSE;

  for (i = 0; i < MaxChromData; i++) {
	  strcpy (Chrom[i].Filename, "");
	  strcpy (Chrom[i].ChrName, "");
	  strcpy (Chrom[i].InfoName, "");
	  strcpy (Chrom[i].LastName, "");
	  Chrom[i].ChromHead = calloc (1, sizeof (struct ChromHeadStruct));
	  Chrom[i].ChromDataArrayActualSize = 20000;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].ChromDataArrays[j] = calloc (Chrom[i].ChromDataArrayActualSize, sizeof (int));
	  Chrom[i].Peak = calloc (1, sizeof (struct PeakStruct));
	  Chrom[i].IstPeak = calloc (1, sizeof (struct PeakStruct));
	  Chrom[i].PeakArrayActualSize = 200;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].PeakArray[j] = calloc (Chrom[i].PeakArrayActualSize, sizeof (struct PeakStruct));
	  Chrom[i].LayerQueueActualSize = 200;
	  bzero (&Chrom[i].QueueHead, sizeof(struct LayerQueueHeadStruct));
	  Chrom[i].QueueItemArray = calloc (Chrom[i].LayerQueueActualSize, sizeof (struct QueueItemStruct));
	  Chrom[i].QueueOrSeriesActive = FALSE;
	  bzero (&Chrom[i].TR541RawDataArray, 200*1000*sizeof(int));
	  Chrom[i].ActualDetector = 0;
	  Chrom[i].ActualExtension = 0;

	  Chrom[i].CalibrHead = calloc (1, sizeof (struct CalibrHeadStruct));
	  bzero (Chrom[i].CalibrHead, sizeof (struct CalibrHeadStruct));
	  Chrom[i].CalibrComponentActualSize = 20;
	  Chrom[i].CalibrComponent = calloc (1, sizeof (struct CalibrComponentStruct));
	  Chrom[i].CalibrComponentArray = calloc (Chrom[i].CalibrComponentActualSize, sizeof (struct CalibrComponentStruct));
	  Chrom[i].CalibrLevelActualSize = 10;
	  Chrom[i].CalibrLevel = calloc (1, sizeof (struct CalibrLevelStruct));
	  Chrom[i].CalibrLevelArray = calloc (Chrom[i].CalibrLevelActualSize, sizeof (struct CalibrLevelStruct));
	  Chrom[i].CalibrChromActualSize = 20;
	  Chrom[i].CalibrChrom = calloc (1, sizeof (struct CalibrChromStruct));
	  Chrom[i].CalibrChromArray = calloc (Chrom[i].CalibrChromActualSize, sizeof (struct CalibrChromStruct));
	  Chrom[i].CalibrCellActualSize = 100;
	  Chrom[i].CalibrCell = calloc (1, sizeof (struct CalibrCellStruct));
	  Chrom[i].CalibrIstCell = calloc (1, sizeof (struct CalibrCellStruct));
	  Chrom[i].CalibrCellArray = calloc (Chrom[i].CalibrCellActualSize, sizeof (struct CalibrCellStruct));
	  Chrom[i].CalibrConcActualSize = 50;
	  Chrom[i].CalibrConc = calloc (1, sizeof (struct CalibrConcStruct));
	  Chrom[i].CalibrConcArray = calloc (Chrom[i].CalibrConcActualSize, sizeof (struct CalibrConcStruct));

	  strcpy (Chrom[i].CalibrTemplateChromName, "");
	  Chrom[i].CalibrTemplateChromHead = calloc (1, sizeof (struct ChromHeadStruct));
	  Chrom[i].CalibrTemplateChromDataArrayActualSize = 20000;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].CalibrTemplateChromDataArrays[j] = calloc (Chrom[i].CalibrTemplateChromDataArrayActualSize, sizeof (int));
	  Chrom[i].CalibrTemplatePeak = calloc (1, sizeof (struct PeakStruct));
	  Chrom[i].CalibrTemplatePeakArrayActualSize = 200;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].CalibrTemplatePeakArray[j] = calloc (Chrom[i].CalibrTemplatePeakArrayActualSize, sizeof (struct PeakStruct));

	  strcpy (Chrom[i].CalibrChromName, "");
	  Chrom[i].CalibrChromHead = calloc (1, sizeof (struct ChromHeadStruct));
	  Chrom[i].CalibrChromDataArrayActualSize = 20000;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].CalibrChromDataArrays[j] = calloc (Chrom[i].CalibrChromDataArrayActualSize, sizeof (int));
	  Chrom[i].CalibrPeak = calloc (1, sizeof (struct PeakStruct));
	  Chrom[i].CalibrPeakArrayActualSize = 200;
	  for (j = 0; j < MaxDetectors; j++)
		  Chrom[i].CalibrPeakArray[j] = calloc (Chrom[i].CalibrPeakArrayActualSize, sizeof (struct PeakStruct));
  } // i

  LastChromDir = calloc (FilenameLength, sizeof (char));
  LastMainOpenDir = calloc (FilenameLength, sizeof (char));

  EventName = calloc (200, sizeof (char));
  for (i = 0; i < EventNumbers; i++)
	  EventNames[i] = calloc (200, sizeof (char));

  bzero (&ADRawValue, sizeof (ADRawValue));
  bzero (&ADRawResponse, sizeof (ADRawResponse));
  bzero (&ADRawFreshData, sizeof (ADRawFreshData));
  bzero (&ADRawINPort, sizeof (ADRawINPort));
  bzero (&ADRawOUTPort, sizeof (ADRawOUTPort));
  bzero (&ADFreshOUTFlag, sizeof (ADRawOUTPort));
  bzero (&ADRawTimeOut, sizeof (ADRawTimeOut));

  ADTraceChannel = 0;
  ADTraceHeartBeat = 0;
  ADTraceResponse = 0;
  ADTraceFreshData = 0;
  ADTraceData = 0;
  ADTraceInput = 0;
  ADTraceOutput = 0;
  ADTraceGradient = 0;
  ADTraceStatus = 0;
  ADTraceStatusGetNow = 0;
  bzero (&ADTraceString, ADTraceStringLength);

  bzero (&MeasureFreshDataFlag, sizeof (MeasureFreshDataFlag));
  bzero (&MeasureDataValue, sizeof (MeasureDataValue));
  bzero (&MeasureScaleValue, sizeof (MeasureScaleValue));
  bzero (&MeasureResponseValue, sizeof (MeasureResponseValue));
  bzero (&MeasureErrorFlag, sizeof (MeasureErrorFlag));

  pthread_mutex_init (&bug_report_mutex, NULL);
  pthread_mutex_init (&ad_trace_mutex, NULL);
} // SystemInit

extern void
SystemDone (void) {
int i,j;

  pthread_mutex_destroy (&ad_trace_mutex);
  pthread_mutex_destroy (&bug_report_mutex);

  free (EventName);
  for (i = 0; i < EventNumbers; i++)
	  free (EventNames[i]);
  free (LastMainOpenDir);
  free (LastChromDir);

  for (i = 0; i < MaxChromData; i++) {
	  free (Chrom[i].CalibrHead);
	  free (Chrom[i].CalibrComponent);
	  free (Chrom[i].CalibrComponentArray);
	  free (Chrom[i].CalibrLevel);
	  free (Chrom[i].CalibrLevelArray);
	  free (Chrom[i].CalibrChrom);
	  free (Chrom[i].CalibrChromArray);
	  free (Chrom[i].CalibrCell);
	  free (Chrom[i].CalibrIstCell);
	  free (Chrom[i].CalibrCellArray);
	  free (Chrom[i].CalibrConc);
	  free (Chrom[i].CalibrConcArray);

	  free (Chrom[i].ChromHead);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].ChromDataArrays[j]);
	  free (Chrom[i].Peak);
	  free (Chrom[i].IstPeak);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].PeakArray[j]);
	  free (Chrom[i].QueueItemArray);

	  free (Chrom[i].CalibrTemplateChromHead);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].CalibrTemplateChromDataArrays[j]);
	  free (Chrom[i].CalibrTemplatePeak);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].CalibrTemplatePeakArray[j]);

	  free (Chrom[i].CalibrChromHead);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].CalibrChromDataArrays[j]);
	  free (Chrom[i].CalibrPeak);
	  for (j = 0; j < MaxDetectors; j++)
		  free (Chrom[i].CalibrPeakArray[j]);
  } // i

  free (ImportScaleDim);
  free (ImportComment);
  free (ImportLastDir);

  free (SpectraName);

  free (BugTraceCentralDir);
  free (BugTraceDir);

  free (ASCIITime);
  free (ConfigInternalFieldName);
  free (ConfigInternalName);
} // SystemDone

extern void
SystemTest (void) {
int TestValue;

  printf ("System test init-----------------------------\n");
  printf ("\n");
  printf ("### sizeof (struct PeakStruct) = %d\n", sizeof (struct PeakStruct));
  printf ("### sizeof (struct MeasureParametersStruct) = %d\n", sizeof (struct MeasureParametersStruct));
  printf ("### sizeof (struct CalibrHeadStruct) = %d\n", sizeof (struct CalibrHeadStruct));
  printf ("\n");
  printf ("System test done.---------------------------\n");
} // SystemTest
